#include "wifi_board.h"
#include "codecs/es8311_audio_codec.h"
#include "display/lcd_display.h"
#include "application.h"
#include "button.h"
#include "config.h"
#include "led/single_led.h"
#include "assets/lang_config.h"
#include <esp_log.h>
#include <esp_efuse_table.h>
#include <driver/i2c_master.h>
#include "esp_lcd_ili9341.h"
#include <esp_lcd_panel_io.h>
#include <esp_lcd_panel_ops.h>
#include "lamp_controller.h"

#include "system_reset.h"
#include "driver/gpio.h"
#include "driver/spi_master.h"
#include <esp_timer.h>
#include "i2c_device.h"
#include <esp_lcd_panel_vendor.h>
#include <driver/spi_common.h>
#include "power_save_timer.h"
#include <esp_sleep.h>
#include <driver/rtc_io.h>
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "power_manager.h"
#include "esp_system.h"

#define TAG "ESPS3_28_A"

LV_FONT_DECLARE(font_puhui_16_4);
LV_FONT_DECLARE(font_awesome_16_4);

class Ft6336 : public I2cDevice {
public:
    struct TouchPoint_t {
        int num = 0;
        int x = -1;
        int y = -1;
    };
    
    Ft6336(i2c_master_bus_handle_t i2c_bus, uint8_t addr) : I2cDevice(i2c_bus, addr) {
        uint8_t chip_id = ReadReg(0xA3);
        ESP_LOGI(TAG, "Get chip ID: 0x%02X", chip_id);
        last_chip_id_ = chip_id;
        read_buffer_ = new uint8_t[6];
    }
    ~Ft6336() {
        if (read_buffer_) {
            delete[] read_buffer_;
            read_buffer_ = nullptr;
        }
    }

    void UpdateTouchPoint() {
        if (!read_buffer_) return;
        ReadRegs(0x02, read_buffer_, 6);
        if (read_buffer_[0] == 0xFF) {
            read_buffer_[0] = 0x00;
        }
        tp_.num = read_buffer_[0] & 0x01;
        tp_.x = ((read_buffer_[1] & 0x0F) << 8) | read_buffer_[2];
        tp_.y = ((read_buffer_[3] & 0x0F) << 8) | read_buffer_[4];
    }

    const TouchPoint_t& GetTouchPoint() const {
        return tp_;
    }
   
    static bool Probe(i2c_master_bus_handle_t i2c_bus, uint8_t addr, uint8_t& chip_id) {
        if (!i2c_bus) return false;
        i2c_master_dev_handle_t dev = nullptr;
        i2c_device_config_t cfg = {
            .dev_addr_length = I2C_ADDR_BIT_LEN_7,
            .device_address = addr,
            .scl_speed_hz = 400 * 1000,
            .scl_wait_us = 0,
            .flags = {
                .disable_ack_check = 0,
            },
        };
        esp_err_t ret = i2c_master_bus_add_device(i2c_bus, &cfg, &dev);
        if (ret != ESP_OK || dev == nullptr) {
            return false;
        }
        uint8_t reg = 0xA3;
        uint8_t id = 0;
        ret = i2c_master_transmit_receive(dev, &reg, 1, &id, 1, 100);
        i2c_master_bus_rm_device(dev);
        if (ret == ESP_OK) {
            chip_id = id;
            return true;
        }
        return false;
    }

private:
    uint8_t* read_buffer_ = nullptr;
    TouchPoint_t tp_;
    uint8_t last_chip_id_ = 0;
};

class ESPS3_28_A : public WifiBoard {
private:
    i2c_master_bus_handle_t codec_i2c_bus_ = nullptr; 
    Button boot_button_;
    LcdDisplay* display_;
     //Display* display_ = nullptr;
    esp_timer_handle_t touchpad_timer_ = nullptr;
    Ft6336* ft6336_ = nullptr;
    // Ft6336* ft6336_;
    PowerSaveTimer* power_save_timer_ = nullptr;
    esp_lcd_panel_handle_t panel_ = nullptr;
    PowerManager* power_manager_ = nullptr;

    void InitializePowerSaveTimer() {
        rtc_gpio_init(GPIO_NUM_3);
        rtc_gpio_set_direction(GPIO_NUM_3, RTC_GPIO_MODE_OUTPUT_ONLY);
        rtc_gpio_set_level(GPIO_NUM_3, 1);

        power_save_timer_ = new PowerSaveTimer(-1, 60, 290);
        ESP_LOGI(TAG, "Reset reason: %d", (int)esp_reset_reason());
        power_save_timer_->OnEnterSleepMode([this]() {
           GetDisplay()->SetPowerSaveMode(true);
           GetBacklight()->SetBrightness(1);
        });
        power_save_timer_->OnExitSleepMode([this]() {
           GetDisplay()->SetPowerSaveMode(false);
           GetBacklight()->RestoreBrightness();
        });
        power_save_timer_->OnShutdownRequest([this]() {
           ESP_LOGI(TAG, "Shutting down");
         //   关闭ES8311音频编解码器
           auto codec = GetAudioCodec();
           if (codec) {
               codec->EnableInput(false);
               codec->EnableOutput(false);
           }
           rtc_gpio_set_level(GPIO_NUM_3, 0);
          //  启用保持功能，确保睡眠期间电平不变
           rtc_gpio_hold_en(GPIO_NUM_3);
           esp_lcd_panel_disp_on_off(panel_, false); //关闭显示
           esp_deep_sleep_start();
        });
        power_save_timer_->SetEnabled(false);
    }

    void InitializePowerManager() {
        power_manager_ = new PowerManager(BATTERY_CHARGING_PIN, ADC_CHANNEL_0);
        power_manager_->OnChargingStatusChanged([this](bool is_charging) {
            if (is_charging) {
                power_save_timer_->SetEnabled(false);
            } else {
                power_save_timer_->SetEnabled(true);
            }
        });
     }

    void InitializeCodecI2c() {
        // Initialize I2C peripheral for both audio codec and touch controller
        i2c_master_bus_config_t i2c_bus_cfg = {
            .i2c_port = I2C_NUM_0,
            .sda_io_num = AUDIO_CODEC_I2C_SDA_PIN, 
            .scl_io_num = AUDIO_CODEC_I2C_SCL_PIN,  
            .clk_source = I2C_CLK_SRC_DEFAULT,
        };
        esp_err_t ret = i2c_new_master_bus(&i2c_bus_cfg, &codec_i2c_bus_);
        if (ret != ESP_OK) {
            ESP_LOGE(TAG, "i2c_new_master_bus failed: %s", esp_err_to_name(ret));
            codec_i2c_bus_ = nullptr;
        }
    }
        void PollTouchpad() {
        static bool was_touched = false;
        static int64_t touch_start_time = 0;
        const int64_t TOUCH_THRESHOLD_MS = 500;  // 触摸时长阈值，超过500ms视为长按
        
        if(ft6336_) {
            ft6336_->UpdateTouchPoint();
            auto& touch_point = ft6336_->GetTouchPoint();
            
            // 检测触摸开始
            if (touch_point.num > 0 && !was_touched) {
                was_touched = true;
                touch_start_time = esp_timer_get_time() / 1000; // 转换为毫秒
            } 
            // 检测触摸释放
            else if (touch_point.num == 0 && was_touched) {
                was_touched = false;
                int64_t touch_duration = (esp_timer_get_time() / 1000) - touch_start_time;
                
                // 只有短触才触发
                if (touch_duration < TOUCH_THRESHOLD_MS) {
                    auto& app = Application::GetInstance();
                    if (app.GetDeviceState() == kDeviceStateStarting) {
                        EnterWifiConfigMode();
                        return;
                    }
                    app.ToggleChatState();
                }
            }
        }
    }

    void InitializeFt6336TouchPad() {
        ESP_LOGI(TAG, "Init FT6336");
        
        gpio_config_t io_conf = {};
        io_conf.intr_type = GPIO_INTR_DISABLE;
        io_conf.mode = GPIO_MODE_OUTPUT;
        io_conf.pin_bit_mask = (1ULL << TP_PIN_NUM_TP_RST);
        io_conf.pull_down_en = GPIO_PULLDOWN_DISABLE;
        io_conf.pull_up_en = GPIO_PULLUP_DISABLE;
        gpio_config(&io_conf);

        gpio_config_t int_conf = {};
        int_conf.intr_type = GPIO_INTR_DISABLE;
        int_conf.mode = GPIO_MODE_INPUT;
        int_conf.pin_bit_mask = (1ULL << TP_PIN_NUM_TP_INT);
        int_conf.pull_down_en = GPIO_PULLDOWN_DISABLE;
        int_conf.pull_up_en = GPIO_PULLUP_ENABLE;
        gpio_config(&int_conf);

        // 触摸芯片复位序列
        gpio_set_level(TP_PIN_NUM_TP_RST, 0);
        vTaskDelay(pdMS_TO_TICKS(5));
        gpio_set_level(TP_PIN_NUM_TP_RST, 1);
        vTaskDelay(pdMS_TO_TICKS(50));

        // 探测是否存在触摸芯片
        uint8_t chip_id = 0;
        if (!codec_i2c_bus_) {
            ESP_LOGW(TAG, "Shared I2C bus not initialized, skip touch");
            return;
        }
        bool touch_available = Ft6336::Probe(codec_i2c_bus_, 0x38, chip_id);
        if (!touch_available) {
            ESP_LOGW(TAG, "Ft6336 not found, running in non-touch mode");
            return;
        }

        ft6336_ = new Ft6336(codec_i2c_bus_, 0x38);
        
        // 创建定时器，20ms 间隔
        esp_timer_create_args_t timer_args = {
            .callback = [](void* arg) {
                ESPS3_28_A* board = (ESPS3_28_A*)arg;
                board->PollTouchpad();
            },
            .arg = this,
            .dispatch_method = ESP_TIMER_TASK,
            .name = "touchpad_timer",
            .skip_unhandled_events = true,
        };
        
        ESP_ERROR_CHECK(esp_timer_create(&timer_args, &touchpad_timer_));
        ESP_ERROR_CHECK(esp_timer_start_periodic(touchpad_timer_, 20 * 1000));
    }


    // SPI初始化
    void InitializeSpi() {
        spi_bus_config_t buscfg = {};
        buscfg.mosi_io_num = DISPLAY_MOSI_PIN;
        buscfg.miso_io_num = DISPLAY_MISO_PIN;
        buscfg.sclk_io_num = DISPLAY_CLK_PIN;
        buscfg.quadwp_io_num = GPIO_NUM_NC;
        buscfg.quadhd_io_num = GPIO_NUM_NC;
        buscfg.max_transfer_sz = DISPLAY_WIDTH * DISPLAY_HEIGHT * sizeof(uint16_t);
        ESP_ERROR_CHECK(spi_bus_initialize(SPI3_HOST, &buscfg, SPI_DMA_CH_AUTO));
    }

    void InitializeLcdDisplay() {
        esp_lcd_panel_io_handle_t panel_io = nullptr;
        esp_lcd_panel_handle_t panel = nullptr;
        // 液晶屏控制IO初始化
        ESP_LOGD(TAG, "Install panel IO");
        esp_lcd_panel_io_spi_config_t io_config = {};
        io_config.cs_gpio_num = DISPLAY_CS_PIN;
        io_config.dc_gpio_num = DISPLAY_DC_PIN;
         io_config.spi_mode = DISPLAY_SPI_MODE;
        io_config.pclk_hz = 40 * 1000 * 1000;
        io_config.trans_queue_depth = 10;
        io_config.lcd_cmd_bits = 8;
        io_config.lcd_param_bits = 8;
        ESP_ERROR_CHECK(esp_lcd_new_panel_io_spi(SPI3_HOST, &io_config, &panel_io));

        // 初始化液晶屏驱动芯片
        ESP_LOGD(TAG, "Install LCD driver");
        esp_lcd_panel_dev_config_t panel_config = {};
        panel_config.reset_gpio_num = DISPLAY_RST_PIN;
        panel_config.rgb_ele_order = DISPLAY_RGB_ORDER;
        panel_config.bits_per_pixel = 16;
        ESP_ERROR_CHECK(esp_lcd_new_panel_ili9341(panel_io, &panel_config, &panel));

        esp_lcd_panel_reset(panel);
        esp_lcd_panel_init(panel);
        esp_lcd_panel_disp_on_off(panel, true);
        esp_lcd_panel_invert_color(panel, DISPLAY_INVERT_COLOR);
        esp_lcd_panel_swap_xy(panel, DISPLAY_SWAP_XY);
        esp_lcd_panel_mirror(panel, DISPLAY_MIRROR_X, DISPLAY_MIRROR_Y);

        display_ = new SpiLcdDisplay(panel_io, panel,
                                    DISPLAY_WIDTH, DISPLAY_HEIGHT, DISPLAY_OFFSET_X, DISPLAY_OFFSET_Y, DISPLAY_MIRROR_X, DISPLAY_MIRROR_Y, DISPLAY_SWAP_XY);

        panel_ = panel;
    }

    void InitializeButtons() {
        boot_button_.OnClick([this]() {
            auto& app = Application::GetInstance();
            // During startup (before connected), pressing BOOT button enters Wi-Fi config mode without reboot
            if (app.GetDeviceState() == kDeviceStateStarting) {
                EnterWifiConfigMode();
                return;
            }
            app.ToggleChatState();
        });
    }

public:
    ESPS3_28_A() : boot_button_(BOOT_BUTTON_GPIO) {
        // 配置PA_PIN为输出
        gpio_config_t pa_pin_config = {
        .pin_bit_mask = (1ULL << AUDIO_CODEC_PA_PIN),
        .mode = GPIO_MODE_OUTPUT,
        .pull_up_en = GPIO_PULLUP_DISABLE,
        .pull_down_en = GPIO_PULLDOWN_DISABLE,
        .intr_type = GPIO_INTR_DISABLE,
    };
        gpio_config(&pa_pin_config);
        // 在初始化任何组件之前，先将音频使能引脚设置为禁用状态（低电平有效）
        gpio_set_level(AUDIO_CODEC_PA_PIN, 1); // 设置为高电平，禁用音频放大器
    
        // 初始化共享的I2C总线（用于音频和触摸）
        InitializeCodecI2c();
        
        // 在同一个I2C总线上初始化触摸控制器
        InitializeFt6336TouchPad();
        
        // 显示相关先建立起来
        InitializeSpi();
        InitializeLcdDisplay();
        InitializeButtons();

        if (GetBacklight()) {
            GetBacklight()->RestoreBrightness();
        }

        // 确保音频编解码器已准备好，但先不要启用音频输出
        auto audio_codec = GetAudioCodec();
        if (audio_codec) {
            ESP_LOGI(TAG, "Audio codec initialized successfully");
            // 暂时不启用音频输出，直到系统完全就绪
            // audio_codec->EnableOutput(false);
            // audio_codec->EnableInput(false);
        }

        // 在系统完全初始化后，最后才启用音频输出
        if (audio_codec) {
            // 延迟一段时间确保系统稳定后再启用音频
            vTaskDelay(pdMS_TO_TICKS(100)); 
            gpio_set_level(AUDIO_CODEC_PA_PIN, 0); // 现在启用音频放大器
            // audio_codec->EnableOutput(true);
            // audio_codec->EnableInput(true);
        }

        //显示和背光可用后再初始化省电逻辑，避免空指针
        // InitializePowerSaveTimer();
        // InitializePowerManager();
    }
~ ESPS3_28_A() {
        if (touchpad_timer_) {
            esp_timer_stop(touchpad_timer_);
            esp_timer_delete(touchpad_timer_);
            touchpad_timer_ = nullptr;
        }
        if (ft6336_) {
            delete ft6336_;
            ft6336_ = nullptr;
        }
        if (power_save_timer_) {
            delete power_save_timer_;
            power_save_timer_ = nullptr;
        }
        if (power_manager_) {
            delete power_manager_;
            power_manager_ = nullptr;
        }
        if (display_) {
            delete display_;
            display_ = nullptr;
        }
        // if (codec_i2c_bus_) {
        //     i2c_del_master_bus(codec_i2c_bus_);
        //     codec_i2c_bus_ = nullptr;
        // }
        if (codec_i2c_bus_) {
            i2c_del_master_bus(codec_i2c_bus_);
            codec_i2c_bus_ = nullptr;
        }
    }

    virtual Led* GetLed() override {
        static SingleLed led(BUILTIN_LED_GPIO);
        return &led;
    }

    virtual Display* GetDisplay() override {
        return display_;
    }

    virtual Backlight* GetBacklight() override {
        static PwmBacklight backlight(DISPLAY_BACKLIGHT_PIN, DISPLAY_BACKLIGHT_OUTPUT_INVERT);
        return &backlight;
    }

    virtual AudioCodec* GetAudioCodec() override {
        static Es8311AudioCodec audio_codec(codec_i2c_bus_, I2C_NUM_0, AUDIO_INPUT_SAMPLE_RATE, AUDIO_OUTPUT_SAMPLE_RATE,
            AUDIO_I2S_GPIO_MCLK, AUDIO_I2S_GPIO_BCLK, AUDIO_I2S_GPIO_WS, AUDIO_I2S_GPIO_DOUT, AUDIO_I2S_GPIO_DIN,
            AUDIO_CODEC_PA_PIN, AUDIO_CODEC_ES8311_ADDR, true, true); 
        return &audio_codec;
    }


    virtual bool GetBatteryLevel(int& level, bool& charging, bool& discharging) override {
        if (!power_manager_) {
            level = 0;
            charging = false;
            discharging = true;
            return false;
        }
        
        static bool last_discharging = false;
        charging = power_manager_->IsCharging();
        discharging = power_manager_->IsDischarging();
        if (discharging != last_discharging) {
            power_save_timer_->SetEnabled(discharging);
            last_discharging = discharging;
        }
        level = power_manager_->GetBatteryLevel();
        return true;
    }

    // virtual void SetPowerSaveLevel(PowerSaveLevel level) override {
    //     if (level != PowerSaveLevel::LOW_POWER) {
    //         power_save_timer_->WakeUp();
    //     }
    //     WifiBoard::SetPowerSaveLevel(level);
    // }
};

DECLARE_BOARD(ESPS3_28_A);
