/*
 * SPDX-FileCopyrightText: 2024-2025 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Apache-2.0
 */

/*********************
 *      INCLUDES
 *********************/
#include <string.h>
#include <inttypes.h>
#include "esp_err.h"
#include "esp_log.h"
#include "esp_check.h"
#include "common/gfx_comm.h"
#include "core/gfx_refr_priv.h"
#include "widget/gfx_anim.h"
#include "lib/eaf/gfx_eaf_dec.h"

/*********************
 *      DEFINES
 *********************/
/* Use generic type checking macro from gfx_obj_priv.h */
#define CHECK_OBJ_TYPE_ANIMATION(obj) CHECK_OBJ_TYPE(obj, GFX_OBJ_TYPE_ANIMATION, TAG)

/* Palette cache flags - using bit 16 as transparency marker (RGB565 uses only bits 0-15) */
#define GFX_PALETTE_CACHE_UNINITIALIZED  0xFFFFFFFFU  /*!< Uninitialized cache entry */
#define GFX_PALETTE_CACHE_TRANSPARENT    0x00010000U  /*!< Transparency flag bit mask */
#define GFX_PALETTE_CACHE_COLOR_MASK     0x0000FFFFU  /*!< Color value mask (RGB565) */

/* Helper macros for palette cache operations */
#define GFX_PALETTE_IS_TRANSPARENT(cache_val)  ((cache_val) & GFX_PALETTE_CACHE_TRANSPARENT)
#define GFX_PALETTE_GET_COLOR(cache_val)       ((cache_val) & GFX_PALETTE_CACHE_COLOR_MASK)
#define GFX_PALETTE_SET_TRANSPARENT()          (GFX_PALETTE_CACHE_TRANSPARENT)
#define GFX_PALETTE_SET_COLOR(color_val)       ((color_val) & GFX_PALETTE_CACHE_COLOR_MASK)

/**********************
 *      TYPEDEFS
 **********************/

/* Mirror mode enumeration */
typedef enum {
    GFX_MIRROR_DISABLED = 0,     /*!< Mirror disabled */
    GFX_MIRROR_MANUAL = 1,       /*!< Manual mirror with fixed offset */
    GFX_MIRROR_AUTO = 2          /*!< Auto mirror with calculated offset */
} gfx_mirror_mode_t;

/* Frame processing information structure */
typedef struct {
    /*!< Pre-parsed header information to avoid repeated parsing */
    eaf_header_t header;           /*!< Pre-parsed header for current frame */

    /*!< Pre-fetched frame data to avoid repeated fetching */
    const void *frame_data;          /*!< Pre-fetched frame data for current frame */
    size_t frame_size;               /*!< Size of pre-fetched frame data */

    /*!< Pre-allocated parsing resources to avoid repeated allocation */
    uint32_t *block_offsets;         /*!< Pre-allocated block offsets array */
    uint8_t *pixel_buffer;           /*!< Pre-allocated pixel decode buffer */
    uint32_t *color_palette;         /*!< Pre-allocated color palette cache */

    /*!< Decoding state tracking */
    int last_block;                  /*!< Last decoded block index to avoid repeated decoding */
} gfx_anim_frame_info_t;

/* Animation context structure */
typedef struct {
    uint32_t start_frame;            /*!< Start frame index */
    uint32_t end_frame;              /*!< End frame index */
    uint32_t current_frame;          /*!< Current frame index */
    uint32_t fps;                    /*!< Frames per second */
    bool is_playing;                 /*!< Whether animation is currently playing */
    bool repeat;                     /*!< Whether animation should repeat */
    gfx_timer_handle_t timer;        /*!< Timer handle for frame updates */

    /*!< Frame processing information */
    eaf_format_handle_t file_desc;      /*!< Animation file descriptor */
    gfx_anim_frame_info_t frame;     /*!< Frame processing info */

    /*!< Widget-specific display properties */
    gfx_mirror_mode_t mirror_mode;   /*!< Mirror mode */
    int16_t mirror_offset;          /*!< Mirror buffer offset for positioning */
} gfx_anim_t;

/* Bit depth enumeration for renderer selection */
typedef enum {
    GFX_ANIM_DEPTH_4BIT = 4,   /*!< 4-bit color depth */
    GFX_ANIM_DEPTH_8BIT = 8,   /*!< 8-bit color depth */
    GFX_ANIM_DEPTH_24BIT = 24, /*!< 24-bit color depth */
    GFX_ANIM_DEPTH_MAX = 3     /*!< Maximum number of depth types */
} gfx_anim_depth_t;

/* Function pointer type for pixel renderers */
typedef void (*gfx_anim_pixel_renderer_cb_t)(
    gfx_color_t *dest_buf, gfx_coord_t dest_stride,
    const uint8_t *src_buf, gfx_coord_t src_stride,
    const eaf_header_t *header, uint32_t *palette_cache,
    gfx_area_t *clip_area, bool swap,
    gfx_mirror_mode_t mirror_mode, int16_t mirror_offset, int dest_x_offset);

/**********************
 *  STATIC VARIABLES
 **********************/
static const char *TAG = "gfx_anim";

/**********************
 *  STATIC PROTOTYPES
 **********************/

/* Virtual functions */
static void gfx_draw_animation(gfx_obj_t *obj, int x1, int y1, int x2, int y2, const void *dest_buf, bool swap);
static esp_err_t gfx_anim_delete(gfx_obj_t *obj);

/* Frame management functions */
static void free_frame_buffers(gfx_anim_frame_info_t *frame);
static void gfx_anim_reset_frame(gfx_anim_frame_info_t *frame);
static esp_err_t gfx_anim_prepare_frame(gfx_obj_t *obj);

/* Rendering functions */
static esp_err_t gfx_anim_render_pixels(uint8_t bit_depth,
                                        gfx_color_t *dest_pixels, gfx_coord_t dest_stride,
                                        const uint8_t *src_pixels, gfx_coord_t src_stride,
                                        const eaf_header_t *header, uint32_t *palette_cache,
                                        gfx_area_t *clip_area, bool swap,
                                        gfx_mirror_mode_t mirror_mode, int16_t mirror_offset, int dest_x_offset);

static void gfx_anim_render_4bit_pixels(gfx_color_t *dest_pixels, gfx_coord_t dest_stride,
                                        const uint8_t *src_pixels, gfx_coord_t src_stride,
                                        const eaf_header_t *header, uint32_t *palette_cache,
                                        gfx_area_t *clip_area, bool swap,
                                        gfx_mirror_mode_t mirror_mode, int16_t mirror_offset, int dest_x_offset);

static void gfx_anim_render_8bit_pixels(gfx_color_t *dest_pixels, gfx_coord_t dest_stride,
                                        const uint8_t *src_pixels, gfx_coord_t src_stride,
                                        const eaf_header_t *header, uint32_t *palette_cache,
                                        gfx_area_t *clip_area, bool swap,
                                        gfx_mirror_mode_t mirror_mode, int16_t mirror_offset, int dest_x_offset);

static void gfx_anim_render_24bit_pixels(gfx_color_t *dest_pixels, gfx_coord_t dest_stride,
        const uint8_t *src_pixels, gfx_coord_t src_stride,
        const eaf_header_t *header, uint32_t *palette_cache,
        gfx_area_t *clip_area, bool swap,
        gfx_mirror_mode_t mirror_mode, int16_t mirror_offset, int dest_x_offset);

/* Timer callback */
static void gfx_anim_timer_callback(void *arg);

/**********************
 *  STATIC VARIABLES
 **********************/

/* Renderer function table */
static const gfx_anim_pixel_renderer_cb_t g_anim_renderers[GFX_ANIM_DEPTH_MAX] = {
    gfx_anim_render_4bit_pixels,     // GFX_ANIM_DEPTH_4BIT = 4 (index 0)
    gfx_anim_render_8bit_pixels,     // GFX_ANIM_DEPTH_8BIT = 8 (index 1)
    gfx_anim_render_24bit_pixels,    // GFX_ANIM_DEPTH_24BIT = 24 (index 2)
};

/**********************
 *   STATIC FUNCTIONS
 **********************/

/*=====================
 * Frame Management Functions
 *====================*/

/**
 * @brief Free allocated buffers for frame processing
 */
static void free_frame_buffers(gfx_anim_frame_info_t *frame)
{
    if (frame->block_offsets) {
        free(frame->block_offsets);
        frame->block_offsets = NULL;
    }
    if (frame->pixel_buffer) {
        free(frame->pixel_buffer);
        frame->pixel_buffer = NULL;
    }
    if (frame->color_palette) {
        free(frame->color_palette);
        frame->color_palette = NULL;
    }
}

/**
 * @brief Reset frame info and free all associated resources
 */
static void gfx_anim_reset_frame(gfx_anim_frame_info_t *frame)
{
    if (frame->header.width > 0) {
        eaf_free_header(&frame->header);
        memset(&frame->header, 0, sizeof(eaf_header_t));
    }

    free_frame_buffers(frame);

    frame->frame_data = NULL;
    frame->frame_size = 0;
    frame->last_block = -1;
}

/**
 * @brief Prepare animation frame and update object properties
 */
static esp_err_t gfx_anim_prepare_frame(gfx_obj_t *obj)
{
    esp_err_t ret = ESP_OK;

    gfx_anim_t *anim = (gfx_anim_t *)obj->src;
    uint32_t current_frame = anim->current_frame;
    eaf_format_handle_t file_desc = anim->file_desc;

    eaf_format_type_t frame_format = eaf_probe_frame_info(file_desc, current_frame);
    if (frame_format != EAF_FORMAT_VALID) {
        ESP_LOGE(TAG, "Invalid EAF format for frame %" PRIu32 ": %d", current_frame, frame_format);
        return ESP_FAIL;
    }

    gfx_anim_reset_frame(&anim->frame);

    const void *frame_data = eaf_get_frame_data(file_desc, current_frame);
    size_t frame_size = eaf_get_frame_size(file_desc, current_frame);
    ESP_RETURN_ON_FALSE(frame_data != NULL, ESP_FAIL, TAG, "Frame %" PRIu32 " data unavailable", current_frame);

    anim->frame.frame_data = frame_data;
    anim->frame.frame_size = frame_size;

    eaf_format_type_t format = eaf_get_frame_info(file_desc, current_frame, &anim->frame.header);
    if (format == EAF_FORMAT_FLAG) {
        return ESP_FAIL;
    } else if (format == EAF_FORMAT_INVALID) {
        ESP_GOTO_ON_FALSE(false, ESP_ERR_INVALID_STATE, err, TAG, "Invalid EAF format for frame %lu", current_frame);
    }

    const eaf_header_t *header = &anim->frame.header;
    int frame_width = header->width;
    int num_blocks = header->blocks;
    int block_height = header->block_height;
    uint8_t bit_depth = header->bit_depth;

    anim->frame.block_offsets = (uint32_t *)malloc(num_blocks * sizeof(uint32_t));
    ESP_GOTO_ON_FALSE(anim->frame.block_offsets != NULL, ESP_ERR_NO_MEM, err, TAG, "No mem for block offsets");

    size_t pixel_buffer_size;
    if (bit_depth == 4) {
        pixel_buffer_size = frame_width * (block_height + (block_height % 2)) / 2;
        anim->frame.pixel_buffer = (uint8_t *)malloc(pixel_buffer_size);
    } else if (bit_depth == 8) {
        pixel_buffer_size = frame_width * block_height;
        anim->frame.pixel_buffer = (uint8_t *)malloc(pixel_buffer_size);
    } else if (bit_depth == 24) {
        pixel_buffer_size = frame_width * block_height * 2;
        anim->frame.pixel_buffer = (uint8_t *)heap_caps_aligned_alloc(16, pixel_buffer_size, MALLOC_CAP_DEFAULT);
    } else {
        ESP_GOTO_ON_FALSE(false, ESP_ERR_INVALID_ARG, err, TAG, "Unsupported bit depth: %d", bit_depth);
    }
    ESP_GOTO_ON_FALSE(anim->frame.pixel_buffer != NULL, ESP_ERR_NO_MEM, err, TAG, "No mem for pixel buffer");

    /* Allocate color palette for indexed color modes */
    uint16_t palette_size = 0;
    if (bit_depth == 4) {
        palette_size = 16;
    } else if (bit_depth == 8) {
        palette_size = 256;
    }

    if (palette_size > 0) {
        anim->frame.color_palette = (uint32_t *)heap_caps_malloc(palette_size * sizeof(uint32_t), MALLOC_CAP_INTERNAL | MALLOC_CAP_8BIT);
        ESP_GOTO_ON_FALSE(anim->frame.color_palette != NULL, ESP_ERR_NO_MEM, err, TAG, "No mem for color palette");

        /* Initialize palette cache to uninitialized state */
        memset(anim->frame.color_palette, 0xFF, palette_size * sizeof(uint32_t));
    }

    eaf_calculate_offsets(header, anim->frame.block_offsets);

    obj->geometry.width = header->width;
    obj->geometry.height = header->height;

    /* Get parent screen size for mirror offset calculation */
    uint32_t parent_w, parent_h;
    gfx_emote_get_screen_size(obj->parent_handle, &parent_w, &parent_h);

    /* Calculate mirror offset */
    uint32_t mirror_offset = 0;
    if (anim->mirror_mode == GFX_MIRROR_AUTO) {
        mirror_offset = parent_w - ((obj->geometry.width + obj->geometry.x) * 2);
    } else if (anim->mirror_mode == GFX_MIRROR_MANUAL) {
        mirror_offset = anim->mirror_offset;
    }

    if (anim->mirror_mode != GFX_MIRROR_DISABLED) {
        obj->geometry.width = obj->geometry.width * 2 + mirror_offset;
    }

    ESP_LOGD(TAG, "Frame %" PRIu32 " prepared", current_frame);
    return ret;

err:
    free_frame_buffers(&anim->frame);
    return ret;
}

/*=====================
 * Rendering Functions
 *====================*/

/**
 * @brief Render pixels using registered renderer
 */
static esp_err_t gfx_anim_render_pixels(uint8_t bit_depth,
                                        gfx_color_t *dest_pixels, gfx_coord_t dest_stride,
                                        const uint8_t *src_pixels, gfx_coord_t src_stride,
                                        const eaf_header_t *header, uint32_t *palette_cache,
                                        gfx_area_t *clip_area, bool swap,
                                        gfx_mirror_mode_t mirror_mode, int16_t mirror_offset, int dest_x_offset)
{
    int renderer_idx;
    switch (bit_depth) {
    case 4:  renderer_idx = 0; break;
    case 8:  renderer_idx = 1; break;
    case 24: renderer_idx = 2; break;
    default:
        ESP_RETURN_ON_FALSE(false, ESP_ERR_INVALID_ARG, TAG, "Unsupported bit depth: %d", bit_depth);
    }

    g_anim_renderers[renderer_idx](dest_pixels, dest_stride, src_pixels, src_stride,
                                   header, palette_cache, clip_area, swap,
                                   mirror_mode, mirror_offset, dest_x_offset);

    return ESP_OK;
}

/**
 * @brief Render 4-bit pixels directly to destination buffer
 */
static void gfx_anim_render_4bit_pixels(gfx_color_t *dest_pixels, gfx_coord_t dest_stride,
                                        const uint8_t *src_pixels, gfx_coord_t src_stride,
                                        const eaf_header_t *header, uint32_t *palette_cache,
                                        gfx_area_t *clip_area, bool swap,
                                        gfx_mirror_mode_t mirror_mode, int16_t mirror_offset, int dest_x_offset)
{
    int width = header->width;
    int clip_width = clip_area->x2 - clip_area->x1;
    int clip_height = clip_area->y2 - clip_area->y1;

    if (mirror_mode == GFX_MIRROR_AUTO) {
        mirror_offset = (dest_stride - (src_stride + dest_x_offset) * 2);
    }

    gfx_color_t color;
    for (int y = 0; y < clip_height; y++) {
        for (int x = 0; x < clip_width; x += 2) {
            uint8_t packed_gray = src_pixels[y * src_stride / 2 + (x / 2)];
            uint8_t index1 = (packed_gray & 0xF0) >> 4;
            uint8_t index2 = (packed_gray & 0x0F);

            if (palette_cache[index1] == GFX_PALETTE_CACHE_UNINITIALIZED) {
                bool is_transparent = eaf_palette_get_color(header, index1, swap, &color);
                if (is_transparent) {
                    palette_cache[index1] = GFX_PALETTE_SET_TRANSPARENT();
                } else {
                    palette_cache[index1] = GFX_PALETTE_SET_COLOR(color.full);
                }
            }

            /* Skip transparent pixels */
            if (!GFX_PALETTE_IS_TRANSPARENT(palette_cache[index1])) {
                color.full = (uint16_t)GFX_PALETTE_GET_COLOR(palette_cache[index1]);
                dest_pixels[y * dest_stride + x] = color;

                if (mirror_mode != GFX_MIRROR_DISABLED) {
                    int mirror_x = width + mirror_offset + width - 1 - x;

                    if (mirror_x >= 0 && (dest_x_offset + mirror_x) < dest_stride) {
                        dest_pixels[y * dest_stride + mirror_x] = color;
                    }
                }
            }

            if (palette_cache[index2] == GFX_PALETTE_CACHE_UNINITIALIZED) {
                bool is_transparent = eaf_palette_get_color(header, index2, swap, &color);
                if (is_transparent) {
                    palette_cache[index2] = GFX_PALETTE_SET_TRANSPARENT();
                } else {
                    palette_cache[index2] = GFX_PALETTE_SET_COLOR(color.full);
                }
            }

            /* Skip transparent pixels */
            if (!GFX_PALETTE_IS_TRANSPARENT(palette_cache[index2])) {
                color.full = (uint16_t)GFX_PALETTE_GET_COLOR(palette_cache[index2]);
                dest_pixels[y * dest_stride + x + 1] = color;

                if (mirror_mode != GFX_MIRROR_DISABLED) {
                    int mirror_x = width + mirror_offset + width - 1 - (x + 1);

                    if (mirror_x >= 0 && (dest_x_offset + mirror_x) < dest_stride) {
                        dest_pixels[y * dest_stride + mirror_x] = color;
                    }
                }
            }
        }
    }
}

/**
 * @brief Render 8-bit pixels directly to destination buffer
 */
static void gfx_anim_render_8bit_pixels(gfx_color_t *dest_pixels, gfx_coord_t dest_stride,
                                        const uint8_t *src_pixels, gfx_coord_t src_stride,
                                        const eaf_header_t *header, uint32_t *palette_cache,
                                        gfx_area_t *clip_area, bool swap,
                                        gfx_mirror_mode_t mirror_mode, int16_t mirror_offset, int dest_x_offset)
{
    int32_t clip_width = clip_area->x2 - clip_area->x1;
    int32_t clip_height = clip_area->y2 - clip_area->y1;
    int32_t width = header->width;

    if (mirror_mode == GFX_MIRROR_AUTO) {
        mirror_offset = (dest_stride - (src_stride + dest_x_offset) * 2);
    }

    gfx_color_t color;
    for (int32_t y = 0; y < clip_height; y++) {
        for (int32_t x = 0; x < clip_width; x++) {
            uint8_t index = src_pixels[y * src_stride + x];
            if (palette_cache[index] == GFX_PALETTE_CACHE_UNINITIALIZED) {
                bool is_transparent = eaf_palette_get_color(header, index, swap, &color);
                if (is_transparent) {
                    palette_cache[index] = GFX_PALETTE_SET_TRANSPARENT();
                } else {
                    palette_cache[index] = GFX_PALETTE_SET_COLOR(color.full);
                }
            }

            /* Skip transparent pixels - no need to write to destination */
            if (!GFX_PALETTE_IS_TRANSPARENT(palette_cache[index])) {
                color.full = (uint16_t)GFX_PALETTE_GET_COLOR(palette_cache[index]);
                dest_pixels[y * dest_stride + x] = color;

                if (mirror_mode != GFX_MIRROR_DISABLED) {
                    int mirror_x = width + mirror_offset + width - 1 - x;

                    if (mirror_x >= 0 && (dest_x_offset + mirror_x) < dest_stride) {
                        dest_pixels[y * dest_stride + mirror_x] = color;
                    }
                }
            }
        }
    }
}

/**
 * @brief Render 24-bit pixels directly to destination buffer
 */
static void gfx_anim_render_24bit_pixels(gfx_color_t *dest_pixels, gfx_coord_t dest_stride,
        const uint8_t *src_pixels, gfx_coord_t src_stride,
        const eaf_header_t *header, uint32_t *palette_cache,
        gfx_area_t *clip_area, bool swap,
        gfx_mirror_mode_t mirror_mode, int16_t mirror_offset, int dest_x_offset)
{
    (void)header;
    (void)palette_cache;
    (void)swap;
    int32_t clip_width = clip_area->x2 - clip_area->x1;
    int32_t clip_height = clip_area->y2 - clip_area->y1;
    int32_t width = src_stride;

    if (mirror_mode == GFX_MIRROR_AUTO) {
        mirror_offset = (dest_stride - (src_stride + dest_x_offset) * 2);
    }

    uint16_t *src_pixels_16 = (uint16_t *)src_pixels;
    uint16_t *dest_pixels_16 = (uint16_t *)dest_pixels;

    for (int32_t y = 0; y < clip_height; y++) {
        for (int32_t x = 0; x < clip_width; x++) {
            dest_pixels_16[y * dest_stride + x] = src_pixels_16[y * src_stride + x];

            if (mirror_mode != GFX_MIRROR_DISABLED) {
                int mirror_x = width + mirror_offset + width - 1 - x;

                if (mirror_x >= 0 && (dest_x_offset + mirror_x) < dest_stride) {
                    dest_pixels_16[y * dest_stride + mirror_x] = src_pixels_16[y * src_stride + x];
                }
            }
        }
    }
}

/*=====================
 * Virtual Functions
 *====================*/

/**
 * @brief Virtual draw function for animation widget
 */
static void gfx_draw_animation(gfx_obj_t *obj, int x1, int y1, int x2, int y2, const void *dest_buf, bool swap)
{
    if (obj == NULL || obj->src == NULL) {
        ESP_LOGE(TAG, "Invalid object or source");
        return;
    }
    if (obj->type != GFX_OBJ_TYPE_ANIMATION) {
        ESP_LOGE(TAG, "Object is not an animation type");
        return;
    }

    /* Animation property and validation */
    gfx_anim_t *anim = (gfx_anim_t *)obj->src;
    if (anim->file_desc == NULL) {
        return;
    }

    /* Frame data validation */
    const void *frame_data = anim->frame.frame_data;
    if (frame_data == NULL) {
        return;
    }
    if (anim->frame.header.width <= 0) {
        ESP_LOGE(TAG, "Invalid header for frame %" PRIu32, anim->current_frame);
        return;
    }

    /* Frame processing resources */
    const eaf_header_t *header = &anim->frame.header;
    uint8_t *pixel_buffer = anim->frame.pixel_buffer;
    uint32_t *block_offsets = anim->frame.block_offsets;
    uint32_t *palette_cache = anim->frame.color_palette;
    int *last_block_idx = &anim->frame.last_block;
    if (block_offsets == NULL || pixel_buffer == NULL) {
        ESP_LOGE(TAG, "Parsing resources not ready for frame %" PRIu32, anim->current_frame);
        return;
    }

    /* Frame dimensions */
    int frame_width = header->width;
    int frame_height = header->height;
    int block_height = header->block_height;
    int num_blocks = header->blocks;

    /* Get parent dimensions and calculate aligned position */
    gfx_obj_calc_pos_in_parent(obj);

    /* Calculate clipping area for object */
    gfx_area_t render_area = {x1, y1, x2, y2};
    gfx_area_t obj_area = {obj->geometry.x, obj->geometry.y, obj->geometry.x + obj->geometry.width, obj->geometry.y + obj->geometry.height};
    gfx_area_t clip_area;

    if (!gfx_area_intersect(&clip_area, &render_area, &obj_area)) {
        return;
    }

    /* Process animation blocks */
    for (int block_idx = 0; block_idx < num_blocks; block_idx++) {
        /* Calculate block boundaries in frame coordinates */
        int block_start_y = block_idx * block_height;
        int block_end_y = (block_idx == num_blocks - 1) ? frame_height : (block_idx + 1) * block_height;
        int block_start_x = 0;
        int block_end_x = frame_width;

        /* Translate to screen coordinates */
        block_start_y += obj->geometry.y;
        block_end_y += obj->geometry.y;
        block_start_x += obj->geometry.x;
        block_end_x += obj->geometry.x;

        /* Calculate clipping area for this block */
        gfx_area_t block_area = {block_start_x, block_start_y, block_end_x, block_end_y};
        gfx_area_t clip_block;

        if (!gfx_area_intersect(&clip_block, &clip_area, &block_area)) {
            continue;
        }

        /* Calculate source buffer offset */
        int src_offset_x = clip_block.x1 - block_start_x;
        int src_offset_y = clip_block.y1 - block_start_y;

        if (src_offset_x < 0 || src_offset_y < 0 ||
                src_offset_x >= frame_width || src_offset_y >= block_height) {
            continue;
        }

        /* Decode block if needed */
        if (block_idx != *last_block_idx) {
            const uint8_t *block_data = (const uint8_t *)frame_data + block_offsets[block_idx];
            int block_len = header->block_len[block_idx];

            esp_err_t decode_result = eaf_decode_block(header, block_data, block_len, pixel_buffer, swap);
            if (decode_result != ESP_OK) {
                continue;
            }
            *last_block_idx = block_idx;
        }

        /* Calculate buffer strides */
        gfx_coord_t dest_stride = (x2 - x1);
        gfx_coord_t src_stride = frame_width;

        /* Calculate source pixel pointer based on bit depth */
        uint8_t *src_pixels = NULL;
        if (header->bit_depth == 24) {
            /* 24-bit depth stored as RGB565 format */
            src_pixels = GFX_BUFFER_OFFSET_16BPP(pixel_buffer,
                                                 src_offset_y,
                                                 src_stride,
                                                 src_offset_x);
        } else if (header->bit_depth == 4) {
            /* 4-bit depth: 2 pixels per byte */
            src_pixels = GFX_BUFFER_OFFSET_4BPP(pixel_buffer,
                                                src_offset_y,
                                                src_stride,
                                                src_offset_x);
        } else {
            /* 8-bit depth: 1 pixel per byte */
            src_pixels = GFX_BUFFER_OFFSET_8BPP(pixel_buffer,
                                                src_offset_y,
                                                src_stride,
                                                src_offset_x);
        }

        /* Calculate destination pixel pointer */
        int dest_x_offset = clip_block.x1 - x1;
        gfx_color_t *dest_pixels = (gfx_color_t *)GFX_BUFFER_OFFSET_16BPP(dest_buf,
                                   clip_block.y1 - y1,
                                   dest_stride,
                                   clip_block.x1 - x1);

        /* Render pixels */
        esp_err_t render_result = gfx_anim_render_pixels(
                                      header->bit_depth,
                                      dest_pixels,
                                      dest_stride,
                                      src_pixels,
                                      src_stride,
                                      header, palette_cache,
                                      &clip_block,
                                      swap,
                                      anim->mirror_mode, anim->mirror_offset, dest_x_offset);

        if (render_result != ESP_OK) {
            continue;
        }
    }
}

/**
 * @brief Virtual delete function for animation widget
 */
static esp_err_t gfx_anim_delete(gfx_obj_t *obj)
{
    CHECK_OBJ_TYPE_ANIMATION(obj);

    gfx_anim_t *anim = (gfx_anim_t *)obj->src;
    if (anim) {
        if (anim->is_playing) {
            gfx_anim_stop(obj);
        }

        if (anim->timer != NULL) {
            gfx_timer_delete((void *)obj->parent_handle, anim->timer);
            anim->timer = NULL;
        }

        gfx_anim_reset_frame(&anim->frame);

        if (anim->file_desc) {
            eaf_deinit(anim->file_desc);
        }

        free(anim);
    }
    return ESP_OK;
}

/*=====================
 * Timer Callback
 *====================*/

/**
 * @brief Timer callback for animation playback
 */
static void gfx_anim_timer_callback(void *arg)
{
    gfx_obj_t *obj = (gfx_obj_t *)arg;
    gfx_anim_t *anim = (gfx_anim_t *)obj->src;

    if (!anim || !anim->is_playing || obj->state.is_visible == false) {
        return;
    }

    gfx_core_context_t *ctx = obj->parent_handle;
    if (anim->current_frame >= anim->end_frame) {
        if (anim->repeat) {
            ESP_LOGD(TAG, "Repeat");
            if (ctx->callbacks.update_cb) {
                ctx->callbacks.update_cb(ctx, GFX_PLAYER_EVENT_ALL_FRAME_DONE, obj);
            }
            anim->current_frame = anim->start_frame;
        } else {
            ESP_LOGD(TAG, "Done");
            anim->is_playing = false;
            if (ctx->callbacks.update_cb) {
                ctx->callbacks.update_cb(ctx, GFX_PLAYER_EVENT_ALL_FRAME_DONE, obj);
            }
            return;
        }
    } else {
        gfx_anim_prepare_frame(obj);
        anim->current_frame++;
        if (ctx->callbacks.update_cb) {
            ctx->callbacks.update_cb(ctx, GFX_PLAYER_EVENT_ONE_FRAME_DONE, obj);
        }
        ESP_LOGD(TAG, "Frame %" PRIu32 "/%" PRIu32, anim->current_frame, anim->end_frame);
    }

    gfx_obj_invalidate(obj);
}

/**********************
 *   PUBLIC FUNCTIONS
 **********************/

/**
 * @brief Create an animation object
 */
gfx_obj_t *gfx_anim_create(gfx_handle_t handle)
{
    gfx_obj_t *obj = (gfx_obj_t *)malloc(sizeof(gfx_obj_t));
    if (obj == NULL) {
        ESP_LOGE(TAG, "No mem for animation object");
        return NULL;
    }

    memset(obj, 0, sizeof(gfx_obj_t));
    obj->type = GFX_OBJ_TYPE_ANIMATION;
    obj->parent_handle = handle;
    obj->state.is_visible = true;
    obj->vfunc.draw = (gfx_obj_draw_fn_t)gfx_draw_animation;
    obj->vfunc.delete = gfx_anim_delete;

    gfx_anim_t *anim = (gfx_anim_t *)malloc(sizeof(gfx_anim_t));
    if (anim == NULL) {
        ESP_LOGE(TAG, "No mem for animation property");
        free(obj);
        return NULL;
    }
    memset(anim, 0, sizeof(gfx_anim_t));

    anim->file_desc = NULL;
    anim->start_frame = 0;
    anim->end_frame = 0;
    anim->current_frame = 0;
    anim->fps = 30;
    anim->repeat = true;
    anim->is_playing = false;

    anim->mirror_mode = GFX_MIRROR_DISABLED;
    anim->mirror_offset = 0;

    uint32_t period_ms = 1000 / anim->fps;
    anim->timer = gfx_timer_create((void *)obj->parent_handle, gfx_anim_timer_callback, period_ms, obj);
    if (anim->timer == NULL) {
        ESP_LOGE(TAG, "Failed to create animation timer");
        free(anim);
        free(obj);
        return NULL;
    }

    memset(&anim->frame.header, 0, sizeof(eaf_header_t));

    anim->frame.frame_data = NULL;
    anim->frame.frame_size = 0;

    anim->frame.block_offsets = NULL;
    anim->frame.pixel_buffer = NULL;
    anim->frame.color_palette = NULL;

    anim->frame.last_block = -1;

    anim->mirror_mode = GFX_MIRROR_DISABLED;
    anim->mirror_offset = 0;

    obj->src = anim;
    obj->type = GFX_OBJ_TYPE_ANIMATION;

    gfx_emote_add_child(handle, GFX_OBJ_TYPE_ANIMATION, obj);
    return obj;
}

/**
 * @brief Set animation source
 */
esp_err_t gfx_anim_set_src(gfx_obj_t *obj, const void *src_data, size_t src_len)
{
    CHECK_OBJ_TYPE_ANIMATION(obj);

    if (src_data == NULL) {
        ESP_LOGE(TAG, "Source data is NULL");
        return ESP_ERR_INVALID_ARG;
    }

    gfx_anim_t *anim = (gfx_anim_t *)obj->src;
    if (anim == NULL) {
        ESP_LOGE(TAG, "Animation property is NULL");
        return ESP_ERR_INVALID_STATE;
    }

    if (anim->is_playing) {
        gfx_anim_stop(obj);
    }

    /* Invalidate the old animation */
    gfx_obj_invalidate(obj);

    if (anim->frame.header.width > 0) {
        eaf_free_header(&anim->frame.header);
        memset(&anim->frame.header, 0, sizeof(eaf_header_t));
    }
    anim->frame.frame_data = NULL;
    anim->frame.frame_size = 0;

    eaf_format_handle_t new_desc;
    eaf_init(src_data, src_len, &new_desc);
    if (new_desc == NULL) {
        ESP_LOGE(TAG, "Failed to initialize asset parser");
        return ESP_FAIL;
    }

    if (anim->file_desc) {
        eaf_deinit(anim->file_desc);
        anim->file_desc = NULL;
    }

    anim->file_desc = new_desc;
    anim->start_frame = 0;
    anim->current_frame = 0;
    anim->end_frame = eaf_get_total_frames(new_desc) - 1;

    gfx_obj_invalidate(obj);

    ESP_LOGD(TAG, "Set src [%" PRIu32 "-%" PRIu32 "]", anim->start_frame, anim->end_frame);
    return ESP_OK;
}

/**
 * @brief Set animation segment
 */
esp_err_t gfx_anim_set_segment(gfx_obj_t *obj, uint32_t start, uint32_t end, uint32_t fps, bool repeat)
{
    CHECK_OBJ_TYPE_ANIMATION(obj);

    gfx_anim_t *anim = (gfx_anim_t *)obj->src;
    if (anim == NULL) {
        ESP_LOGE(TAG, "Animation property is NULL");
        return ESP_ERR_INVALID_STATE;
    }

    int total_frames = eaf_get_total_frames(anim->file_desc);

    anim->start_frame = start;
    anim->end_frame = (end > total_frames - 1) ? (total_frames - 1) : end;
    anim->current_frame = start;

    if (anim->fps != fps) {
        anim->fps = fps;

        if (anim->timer != NULL) {
            uint32_t new_period_ms = 1000 / fps;
            gfx_timer_set_period(anim->timer, new_period_ms);
            ESP_LOGD(TAG, "FPS %" PRIu32 "->%" PRIu32, anim->fps, fps);
        }
    }

    anim->repeat = repeat;

    ESP_LOGD(TAG, "Segment [%" PRIu32 "-%" PRIu32 "] fps:%" PRIu32 " repeat:%d", anim->start_frame, anim->end_frame, fps, repeat);
    return ESP_OK;
}

/**
 * @brief Start animation playback
 */
esp_err_t gfx_anim_start(gfx_obj_t *obj)
{
    CHECK_OBJ_TYPE_ANIMATION(obj);

    gfx_anim_t *anim = (gfx_anim_t *)obj->src;
    if (anim == NULL) {
        ESP_LOGE(TAG, "Animation property is NULL");
        return ESP_ERR_INVALID_STATE;
    }

    if (anim->file_desc == NULL) {
        ESP_LOGE(TAG, "Animation source not set");
        return ESP_ERR_INVALID_STATE;
    }

    if (anim->is_playing) {
        return ESP_OK;
    }

    anim->is_playing = true;
    anim->current_frame = anim->start_frame;

    ESP_LOGD(TAG, "Start");
    return ESP_OK;
}

/**
 * @brief Stop animation playback
 */
esp_err_t gfx_anim_stop(gfx_obj_t *obj)
{
    CHECK_OBJ_TYPE_ANIMATION(obj);

    gfx_anim_t *anim = (gfx_anim_t *)obj->src;
    if (anim == NULL) {
        ESP_LOGE(TAG, "Animation property is NULL");
        return ESP_ERR_INVALID_STATE;
    }

    if (!anim->is_playing) {
        return ESP_OK;
    }

    anim->is_playing = false;

    ESP_LOGD(TAG, "Stop");
    return ESP_OK;
}

/**
 * @brief Set manual mirror mode
 */
esp_err_t gfx_anim_set_mirror(gfx_obj_t *obj, bool enabled, int16_t offset)
{
    CHECK_OBJ_TYPE_ANIMATION(obj);

    gfx_anim_t *anim = (gfx_anim_t *)obj->src;
    if (anim == NULL) {
        ESP_LOGE(TAG, "Animation property is NULL");
        return ESP_ERR_INVALID_STATE;
    }

    anim->mirror_mode = enabled ? GFX_MIRROR_MANUAL : GFX_MIRROR_DISABLED;
    anim->mirror_offset = offset;

    ESP_LOGD(TAG, "Mirror %s offset:%d", enabled ? "on" : "off", offset);
    return ESP_OK;
}

/**
 * @brief Set automatic mirror mode
 */
esp_err_t gfx_anim_set_auto_mirror(gfx_obj_t *obj, bool enabled)
{
    CHECK_OBJ_TYPE_ANIMATION(obj);

    gfx_anim_t *anim = (gfx_anim_t *)obj->src;
    if (anim == NULL) {
        ESP_LOGE(TAG, "Animation property is NULL");
        return ESP_ERR_INVALID_STATE;
    }

    anim->mirror_mode = enabled ? GFX_MIRROR_AUTO : GFX_MIRROR_DISABLED;

    ESP_LOGD(TAG, "Auto mirror %s", enabled ? "on" : "off");
    return ESP_OK;
}
